// ChessNextMove Extension - Side Panel JavaScript
// Handles auth, communication with background, and iframe messaging

// ============================================
// CONSTANTS
// ============================================
const CURRENT_VERSION = "1.1.3";
const VERSION_CHECK_URL = "https://chessnextmove.ai/api/extension-version";

// ============================================
// STATE
// ============================================

let backgroundPort = null;
let calculatorFrame = null;
let isFrameReady = false;
let pendingFen = null;
let currentUser = null;
let authToken = null;
let authCheckInterval = null;
let messageHandler = null; // Track message handler for cleanup

// ============================================
// GLOBAL LOGIN HANDLER (for inline onclick)
// ============================================
async function handleLoginClick(e) {
  if (e) e.preventDefault();
  console.log('[Extension] handleLoginClick called!');
  await doLogin();
}

// ============================================
// DEBUG LOGGING (disabled in production)
// ============================================
const DEBUG = false;
function log(message, data = null) {
  if (!DEBUG) return;
  const timestamp = new Date().toISOString().split('T')[1].split('.')[0];
  if (data) {
    console.log(`[Extension ${timestamp}] ${message}`, data);
  } else {
    console.log(`[Extension ${timestamp}] ${message}`);
  }
}

// ============================================
// UPDATE CHECK
// ============================================
async function checkForUpdates() {
  const banner = document.getElementById('versionBanner');
  const versionText = document.getElementById('versionText');
  
  if (!banner || !versionText) return;
  
  try {
    // Add cache-busting parameter
    const response = await fetch(`${VERSION_CHECK_URL}?t=${Date.now()}`);
    if (!response.ok) {
      versionText.textContent = `v${CURRENT_VERSION}`;
      return;
    }
    
    const data = await response.json();
    const latestVersion = data.version;
    
    if (isNewerVersion(latestVersion, CURRENT_VERSION)) {
      // Update available
      banner.classList.remove('latest');
      banner.classList.add('update-available');
      versionText.textContent = `Update available v${latestVersion} - Click to update`;
      banner.onclick = () => window.open(data.downloadUrl || 'https://chessnextmove.ai/extension-download', '_blank');
    } else {
      // Latest version
      banner.classList.remove('update-available');
      banner.classList.add('latest');
      versionText.textContent = `v${CURRENT_VERSION}`;
      banner.onclick = null;
    }
  } catch (err) {
    log('Update check failed:', err.message);
    versionText.textContent = `v${CURRENT_VERSION}`;
  }
}

function isNewerVersion(latest, current) {
  const latestParts = latest.split('.').map(Number);
  const currentParts = current.split('.').map(Number);
  
  for (let i = 0; i < Math.max(latestParts.length, currentParts.length); i++) {
    const l = latestParts[i] || 0;
    const c = currentParts[i] || 0;
    if (l > c) return true;
    if (l < c) return false;
  }
  return false;
}

// ============================================
// INITIALIZATION
// ============================================

document.addEventListener('DOMContentLoaded', async () => {
  log('DOM loaded, starting initialization...');
  
  // Check for updates
  checkForUpdates();
  
  // IMMEDIATE - attach click handler to login button
  const loginBtn = document.getElementById('loginBtn');
  if (loginBtn) {
    loginBtn.addEventListener('click', async (e) => {
      e.preventDefault();
      e.stopPropagation();
      log('Login button clicked');
      try {
        await doLogin();
      } catch (err) {
        log('doLogin CRASHED:', err.message);
        console.error('Full error:', err);
      }
    }, true);
    log('Login click listener attached');
  }
  
  // Handle Enter key in password field
  const passwordInput = document.getElementById('password');
  if (passwordInput) {
    passwordInput.addEventListener('keydown', async (e) => {
      if (e.key === 'Enter') {
        e.preventDefault();
        log('Enter pressed');
        await doLogin();
      }
    });
  }
  
  // Password visibility toggle
  const togglePassword = document.getElementById('togglePassword');
  if (togglePassword && passwordInput) {
    togglePassword.addEventListener('click', () => {
      const isPassword = passwordInput.type === 'password';
      passwordInput.type = isPassword ? 'text' : 'password';
      document.getElementById('eyeIcon').style.display = isPassword ? 'none' : 'block';
      document.getElementById('eyeOffIcon').style.display = isPassword ? 'block' : 'none';
    });
  }
  
  const { disclaimerAccepted } = await chrome.storage.local.get(['disclaimerAccepted']);
  log('Disclaimer accepted:', disclaimerAccepted);
  
  if (!disclaimerAccepted) {
    document.getElementById('disclaimerPopup').classList.remove('hidden');
    document.getElementById('acceptDisclaimer').addEventListener('click', async () => {
      await chrome.storage.local.set({ disclaimerAccepted: true });
      document.getElementById('disclaimerPopup').classList.add('hidden');
    });
  }
  
  connectToBackground();
  
  const authData = await chrome.storage.local.get(['authToken', 'user', 'tier']);
  log('Stored auth data:', { hasToken: !!authData.authToken, tier: authData.tier });
  
  hideScreen('loadingScreen');
  
  if (!authData.authToken) {
    log('No token found, showing login screen');
    showScreen('loginScreen');
    setupLoginForm();
  } else {
    log('Token found, verifying...');
    const verified = await verifyToken(authData.authToken);
    log('Token verification result:', verified);
    
    if (!verified.valid) {
      log('Token invalid, clearing and showing login');
      await chrome.storage.local.remove(['authToken', 'user', 'tier']);
      showScreen('loginScreen');
      setupLoginForm();
    } else if (verified.tier !== 'pro') {
      log('User is not pro, tier:', verified.tier);
      showScreen('notProScreen');
      document.getElementById('userEmail').textContent = verified.user?.email || authData.user?.email || 'Unknown';
      document.getElementById('userTier').textContent = verified.tier || 'free';
      setupLogoutButton('logoutBtn');
    } else {
      log('User is pro, showing calculator');
      currentUser = verified.user;
      authToken = authData.authToken;
      showScreen('calculatorScreen');
      initializeCalculator();
      startAuthCheck();
    }
  }
});

// ============================================
// SCREEN MANAGEMENT
// ============================================

function showScreen(screenId) {
  log('Showing screen:', screenId);
  document.querySelectorAll('.screen').forEach(s => s.classList.remove('active'));
  const screen = document.getElementById(screenId);
  if (screen) {
    screen.classList.add('active');
  } else {
    log('ERROR: Screen not found:', screenId);
  }
}

function hideScreen(screenId) {
  const screen = document.getElementById(screenId);
  if (screen) {
    screen.classList.remove('active');
  }
}

// ============================================
// BACKGROUND CONNECTION
// ============================================

function connectToBackground() {
  log('=== CONNECTING TO BACKGROUND ===');
  
  try {
    backgroundPort = chrome.runtime.connect({ name: 'sidepanel' });
    log('Background port created:', !!backgroundPort);
    
    backgroundPort.onMessage.addListener((msg) => {
      log('Message from background:', msg.type, msg);
      if (msg.type === 'boardUpdate') handleBoardUpdate(msg.data);
      if (msg.type === 'boardCleared') updateStatus('No board detected', false);
    });
    
    backgroundPort.onDisconnect.addListener(() => {
      log('WARNING: Background port disconnected!');
      log('lastError:', chrome.runtime.lastError?.message);
      backgroundPort = null;
      log('Reconnecting in 1 second...');
      setTimeout(connectToBackground, 1000);
    });
    
    log('Background connection established');
  } catch (err) {
    log('ERROR connecting to background:', err.message);
  }
}

// ============================================
// AUTH
// ============================================

async function verifyToken(token) {
  // Check if online first
  if (!navigator.onLine) {
    log('Offline - skipping token verification');
    return { valid: false, offline: true };
  }
  
  try {
    log('Calling verify API...');
    const response = await fetch('https://chessnextmove.ai/api/auth/mobile/verify', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      }
    });
    
    const data = await response.json();
    log('Verify API response:', data);
    
    if (data.success && data.user) {
      const tier = data.user.tier || 'free';
      await chrome.storage.local.set({ tier: tier });
      return { valid: true, tier: tier, user: data.user };
    }
    
    return { valid: false };
  } catch (err) {
    log('Token verification error:', err.message);
    return { valid: false };
  }
}

function startAuthCheck() {
  if (authCheckInterval) clearInterval(authCheckInterval);
  
  // Try to refresh token on startup
  tryRefreshToken();
  
  authCheckInterval = setInterval(async () => {
    const authData = await chrome.storage.local.get(['authToken']);
    if (!authData.authToken) {
      handleAuthExpired();
      return;
    }
    
    const verified = await verifyToken(authData.authToken);
    if (!verified.valid && !verified.offline) {
      handleAuthExpired();
    } else if (verified.valid && verified.tier !== 'pro') {
      handleAuthExpired();
    }
    
    // Try to refresh token periodically (every 2 minutes check)
    tryRefreshToken();
  }, 120000); // Check every 2 minutes
}

async function tryRefreshToken() {
  if (!navigator.onLine) return;
  
  const authData = await chrome.storage.local.get(['authToken']);
  if (!authData.authToken) return;
  
  try {
    const response = await fetch('https://chessnextmove.ai/api/auth/mobile/refresh', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${authData.authToken}`
      }
    });
    
    const data = await response.json();
    
    if (data.success && data.refreshed && data.token) {
      log('Token refreshed successfully');
      await chrome.storage.local.set({ authToken: data.token });
      authToken = data.token;
    }
  } catch (err) {
    // Silently fail - will retry on next interval
    log('Token refresh failed:', err.message);
  }
}

function handleAuthExpired() {
  log('Auth expired, reloading...');
  if (authCheckInterval) clearInterval(authCheckInterval);
  chrome.storage.local.remove(['authToken', 'user', 'tier']);
  location.reload();
}

function setupLoginForm() {
  log('Setting up login form...');
  const container = document.getElementById('loginForm');
  const btn = document.getElementById('loginBtn');
  
  if (!container || !btn) {
    log('ERROR: Container or button not found!');
    return;
  }
  
  log('Button found, adding click listener');
  
  // Handle button click
  btn.onclick = async (e) => {
    e.preventDefault();
    log('Login button clicked');
    await doLogin();
  };
  
  log('btn.onclick assigned:', typeof btn.onclick);
  
  // Also handle Enter key in password field
  const passwordInput = document.getElementById('password');
  if (passwordInput) {
    passwordInput.onkeydown = async (e) => {
      if (e.key === 'Enter') {
        e.preventDefault();
        log('Enter pressed in password field');
        await doLogin();
      }
    };
  }
  
  log('Login form setup complete');
}

async function doLogin() {
  log('>>> doLogin() STARTED <<<');
  const errorDiv = document.getElementById('loginError');
  const email = document.getElementById('email').value;
  const password = document.getElementById('password').value;
  const btn = document.getElementById('loginBtn');
  
  log('Login attempt for:', email);
  
  if (!email || !password) {
    errorDiv.textContent = 'Please enter email and password';
    return;
  }
  
  // Check if online
  if (!navigator.onLine) {
    errorDiv.textContent = 'No internet connection. Please check your network.';
    return;
  }
  
  btn.disabled = true;
  btn.innerHTML = '<span>Signing in...</span>';
  errorDiv.textContent = '';
  
  try {
    log('Calling login API...');
    const response = await fetch('https://chessnextmove.ai/api/auth/mobile/login', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email, password })
    });
    
    log('Login API response status:', response.status);
    const data = await response.json();
    log('Login API response:', { success: data.success, tier: data.user?.tier, hasToken: !!data.token });
    
    if (data.success && data.token) {
      const tier = data.user?.tier || 'free';
      log('Login successful, tier:', tier);
      
      await chrome.storage.local.set({
        authToken: data.token,
        user: data.user,
        tier: tier
      });
      log('Auth data stored');
      
      if (tier === 'pro') {
        log('User is PRO, switching to calculator');
        currentUser = data.user;
        authToken = data.token;
        hideScreen('loginScreen');
        showScreen('calculatorScreen');
        initializeCalculator();
        startAuthCheck();
      } else {
        log('User is NOT PRO, showing upgrade screen');
        hideScreen('loginScreen');
        showScreen('notProScreen');
        document.getElementById('userEmail').textContent = data.user?.email || email;
        document.getElementById('userTier').textContent = tier;
        setupLogoutButton('logoutBtn');
      }
    } else {
      log('Login failed:', data.error);
      errorDiv.textContent = data.error || 'Login failed';
      btn.disabled = false;
      btn.innerHTML = '<span>Sign In</span>';
    }
  } catch (err) {
    log('Login error:', err.message);
    errorDiv.textContent = 'Network error. Please try again.';
    btn.disabled = false;
    btn.innerHTML = '<span>Sign In</span>';
  }
}

function setupLogoutButton(btnId) {
  const btn = document.getElementById(btnId);
  if (btn) {
    btn.onclick = async () => {
      log('Logout clicked');
      if (authCheckInterval) clearInterval(authCheckInterval);
      await chrome.storage.local.remove(['authToken', 'user', 'tier']);
      location.reload();
    };
  }
}

// ============================================
// CALCULATOR
// ============================================

async function initializeCalculator() {
  log('=== CALCULATOR INIT START ===');
  
  // Re-verify token before loading iframe (catches subscription expiry)
  const authData = await chrome.storage.local.get(['authToken']);
  if (!authData.authToken) {
    log('ERROR: No auth token, showing login');
    handleAuthExpired();
    return;
  }
  
  const verified = await verifyToken(authData.authToken);
  if (!verified.valid) {
    log('Token invalid during calculator init, showing login');
    handleAuthExpired();
    return;
  }
  
  if (verified.tier !== 'pro') {
    log('User no longer pro during calculator init, tier:', verified.tier);
    showScreen('notProScreen');
    document.getElementById('userEmail').textContent = verified.user?.email || 'Unknown';
    document.getElementById('userTier').textContent = verified.tier || 'free';
    setupLogoutButton('logoutBtn');
    return;
  }
  
  // Store token for postMessage auth
  authToken = authData.authToken;
  
  calculatorFrame = document.getElementById('calculatorFrame');
  
  if (!calculatorFrame) {
    log('ERROR: Calculator frame element not found in DOM!');
    return;
  }
  log('Calculator frame element found');
  
  // Load iframe WITHOUT token in URL (security fix)
  const iframeSrc = 'https://chessnextmove.ai/extension';
  log('Setting iframe src to:', iframeSrc);
  
  // Set up load/error handlers BEFORE setting src
  calculatorFrame.onload = () => {
    log('>>> IFRAME ONLOAD FIRED <<<');
    log('Iframe contentWindow exists:', !!calculatorFrame.contentWindow);
    // Auth token is sent via handleFrameMessage when iframe sends 'extensionReady'
    
    if (backgroundPort) {
      log('Requesting board scan from background...');
      backgroundPort.postMessage({ type: 'requestScan' });
    } else {
      log('WARNING: backgroundPort is null, cannot request scan');
    }
  };
  
  calculatorFrame.onerror = (err) => {
    log('>>> IFRAME ONERROR FIRED <<<', err);
  };
  
  // Set the src
  calculatorFrame.src = iframeSrc;
  log('Iframe src SET, waiting for load...');
  
  // Timeout warning if iframe doesn't respond
  setTimeout(() => {
    if (!isFrameReady) {
      log('WARNING: Iframe did not send ready message after 5 seconds!');
      log('isFrameReady:', isFrameReady);
      log('calculatorFrame.src:', calculatorFrame.src ? 'SET' : 'EMPTY');
      log('calculatorFrame.contentWindow:', !!calculatorFrame.contentWindow);
    }
  }, 5000);
  
  // Longer timeout
  setTimeout(() => {
    if (!isFrameReady) {
      log('CRITICAL: Iframe still not ready after 10 seconds - likely black screen issue!');
    }
  }, 10000);
  
  // Clean up previous message handler if exists (fix memory leak)
  if (messageHandler) {
    window.removeEventListener('message', messageHandler);
  }
  messageHandler = handleFrameMessage;
  window.addEventListener('message', messageHandler);
  log('Message listener attached');
  
  const reloadBtn = document.getElementById('reloadBtn');
  if (reloadBtn) {
    reloadBtn.onclick = () => handleReload();
    log('Reload button handler attached');
  }
  
  setupLogoutButton('headerLogoutBtn');
  log('=== CALCULATOR INIT COMPLETE ===');
}

function handleFrameMessage(event) {
  log('>>> MESSAGE RECEIVED from:', event.origin);
  
  if (event.origin !== 'https://chessnextmove.ai') {
    log('Ignoring message from untrusted origin:', event.origin);
    return;
  }
  
  const { type } = event.data;
  log('Frame message type:', type, 'Full data:', event.data);
  
  // Handle extensionReady - iframe is loaded and waiting for auth
  if (type === 'extensionReady') {
    log('>>> EXTENSION READY - Sending auth token <<<');
    if (calculatorFrame && calculatorFrame.contentWindow && authToken) {
      calculatorFrame.contentWindow.postMessage({
        type: 'auth',
        token: authToken
      }, 'https://chessnextmove.ai');
      log('Auth token sent to iframe');
    } else {
      log('WARNING: Cannot send auth - missing frame or token');
    }
  }
  
  if (type === 'ready') {
    log('>>> CALCULATOR READY MESSAGE RECEIVED <<<');
    isFrameReady = true;
    
    if (calculatorFrame && calculatorFrame.contentWindow) {
      log('Sending readyAck to iframe...');
      calculatorFrame.contentWindow.postMessage({ type: 'readyAck' }, 'https://chessnextmove.ai');
    } else {
      log('WARNING: Cannot send readyAck - calculatorFrame or contentWindow is null');
    }
    
    if (pendingFen) {
      log('Sending pending FEN:', pendingFen.fen);
      sendFenToFrame(pendingFen.fen, pendingFen.flipped);
      pendingFen = null;
    }
  }
  
  if (type === 'authExpired') {
    log('Auth expired message from iframe');
    handleAuthExpired();
  }
}

// ============================================
// BOARD UPDATES
// ============================================

function handleBoardUpdate(data) {
  log('Board update:', data);
  updateStatus('Board detected', true);
  
  if (data.fen) {
    if (isFrameReady) {
      sendFenToFrame(data.fen, data.flipped);
    } else {
      pendingFen = { fen: data.fen, flipped: data.flipped };
    }
  }
}

function sendFenToFrame(fen, flipped) {
  if (!calculatorFrame) return;
  
  calculatorFrame.contentWindow.postMessage({
    type: 'loadFen',
    fen: fen,
    flipped: flipped
  }, 'https://chessnextmove.ai');
  
  log('Sent FEN to frame:', fen);
}

function updateStatus(text, connected) {
  const statusBar = document.getElementById('statusBar');
  const statusText = document.getElementById('statusText');
  
  if (statusBar) {
    statusBar.classList.toggle('connected', connected);
    statusBar.classList.remove('scanning');
  }
  
  if (statusText) {
    statusText.textContent = text;
  }
}

// ============================================
// RELOAD - Fix detection issues
// ============================================

function handleReload() {
  log('=== RELOAD BUTTON CLICKED ===');
  log('Current state - isFrameReady:', isFrameReady);
  log('Current state - backgroundPort:', !!backgroundPort);
  log('Current state - calculatorFrame:', !!calculatorFrame);
  
  const reloadBtn = document.getElementById('reloadBtn');
  if (reloadBtn) {
    reloadBtn.classList.add('reloading');
    reloadBtn.disabled = true;
    log('Reload button disabled and spinning');
  }
  
  updateStatus('Reloading...', false);
  
  // Tell background to force re-inject content script on next load
  log('Sending forceReinject message to background...');
  chrome.runtime.sendMessage({ type: 'forceReinject' }, (response) => {
    log('forceReinject response:', response);
    if (chrome.runtime.lastError) {
      log('ERROR sending forceReinject:', chrome.runtime.lastError.message);
    }
    // Reload the sidepanel
    log('Reloading sidepanel in 300ms...');
    setTimeout(() => {
      log('>>> RELOADING NOW <<<');
      location.reload();
    }, 300);
  });
}
