// ChessNextMove Extension - Universal Content Script
// Detects chess boards on ANY website by DOM structure, not URL

(async function() {
  'use strict';
  
  console.log('[ChessNextMove] === CONTENT SCRIPT STARTING ===');
  console.log('[ChessNextMove] URL:', location.href);
  
  // Piece mappings for different board types
  const PIECE_MAP_CLASSES = {
    'wp': 'P', 'wn': 'N', 'wb': 'B', 'wr': 'R', 'wq': 'Q', 'wk': 'K',
    'bp': 'p', 'bn': 'n', 'bb': 'b', 'br': 'r', 'bq': 'q', 'bk': 'k'
  };
  
  const PIECE_MAP_TYPES = {
    'pawn': { 'white': 'P', 'black': 'p' },
    'knight': { 'white': 'N', 'black': 'n' },
    'bishop': { 'white': 'B', 'black': 'b' },
    'rook': { 'white': 'R', 'black': 'r' },
    'queen': { 'white': 'Q', 'black': 'q' },
    'king': { 'white': 'K', 'black': 'k' }
  };
  
  // Generate unique key for this tab using URL
  const TAB_KEY = 'cnm_' + btoa(location.href).substring(0, 20);
  console.log('[ChessNextMove] Storage key:', TAB_KEY);
  
  // Check if already injected using chrome.storage.session
  try {
    console.log('[ChessNextMove] Checking if already injected...');
    const result = await chrome.storage.session.get([TAB_KEY]);
    console.log('[ChessNextMove] Storage result:', result);
    
    if (result[TAB_KEY]) {
      console.log('[ChessNextMove] Already injected! Sending requestScanFromContent message...');
      chrome.runtime.sendMessage({ type: 'requestScanFromContent' }, (response) => {
        console.log('[ChessNextMove] requestScanFromContent response:', response);
        if (chrome.runtime.lastError) {
          console.log('[ChessNextMove] ERROR:', chrome.runtime.lastError.message);
        }
      });
      return;
    }
    
    // Mark as injected
    console.log('[ChessNextMove] Marking as injected in storage...');
    await chrome.storage.session.set({ [TAB_KEY]: true });
    console.log('[ChessNextMove] Storage set complete');
  } catch (err) {
    console.log('[ChessNextMove] Storage check failed:', err.message, '- continuing anyway');
  }
  
  let lastFen = null;
  let observer = null;
  let scanInterval = null;
  let debounceTimer = null;
  
  // Initialize
  function init() {
    console.log('[ChessNextMove] === INIT FUNCTION CALLED ===');
    console.log('[ChessNextMove] document.body exists:', !!document.body);
    
    startWatching();
    
    // Listen for scan requests from background
    chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
      console.log('[ChessNextMove] Message received:', message.type);
      if (message.type === 'scanBoard' || message.type === 'triggerScan') {
        console.log('[ChessNextMove] Triggering scan from message...');
        const result = scanBoard();
        console.log('[ChessNextMove] Scan result:', result);
        sendResponse(result);
      }
      return true;
    });
    
    console.log('[ChessNextMove] Init complete, message listener attached');
  }
  
  // Start watching for board changes
  function startWatching() {
    console.log('[ChessNextMove] === START WATCHING ===');
    
    // Initial scan after short delay
    console.log('[ChessNextMove] Scheduling initial scan in 500ms...');
    setTimeout(() => {
      console.log('[ChessNextMove] Running initial scan...');
      scanBoard();
    }, 500);
    
    // Set up MutationObserver
    console.log('[ChessNextMove] Setting up MutationObserver...');
    observer = new MutationObserver((mutations) => {
      let shouldScan = false;
      for (const mutation of mutations) {
        if (mutation.type === 'childList' || 
            (mutation.type === 'attributes' && 
             (mutation.attributeName === 'class' || mutation.attributeName === 'style'))) {
          shouldScan = true;
          break;
        }
      }
      if (shouldScan) {
        // Debounce: wait 300ms after last mutation to avoid animation frames
        if (debounceTimer) clearTimeout(debounceTimer);
        debounceTimer = setTimeout(() => {
          scanBoard();
        }, 300);
      }
    });
    
    observer.observe(document.body, {
      childList: true,
      subtree: true,
      attributes: true,
      attributeFilter: ['class', 'style']
    });
    console.log('[ChessNextMove] MutationObserver attached to document.body');
    
    // Periodic scan every 2 seconds
    console.log('[ChessNextMove] Starting periodic scan interval (2s)...');
    scanInterval = setInterval(() => scanBoard(), 2000);
  }
  
  // Main scan function - tries multiple board detection methods
  function scanBoard() {
    try {
      let boardData = null;
      
      // Method 1: wc-chess-board (modern web component)
      boardData = scanWcChessBoard();
      if (boardData) {
        console.log('[ChessNextMove] Board found via: wc-chess-board');
      }
      
      // Method 2: chess-board web component
      if (!boardData) {
        boardData = scanChessBoardComponent();
        if (boardData) {
          console.log('[ChessNextMove] Board found via: chess-board component');
        }
      }
      
      // Method 3: cg-board (Lichess-style)
      if (!boardData) {
        boardData = scanCgBoard();
        if (boardData) {
          console.log('[ChessNextMove] Board found via: cg-board');
        }
      }
      
      // Method 4: Generic piece elements with square classes
      if (!boardData) {
        boardData = scanGenericPieces();
        if (boardData) {
          console.log('[ChessNextMove] Board found via: generic pieces');
        }
      }
      
      if (boardData && boardData.fen) {
        if (boardData.fen !== lastFen) {
          lastFen = boardData.fen;
          
          console.log('[ChessNextMove] NEW FEN detected:', boardData.fen, 'Flipped:', boardData.flipped);
          console.log('[ChessNextMove] Sending boardDetected message to background...');
          
          try {
            chrome.runtime.sendMessage({
              type: 'boardDetected',
              fen: boardData.fen,
              flipped: boardData.flipped
            }, (response) => {
              if (chrome.runtime.lastError) {
                console.log('[ChessNextMove] ERROR sending boardDetected:', chrome.runtime.lastError.message);
              } else {
                console.log('[ChessNextMove] boardDetected message sent, response:', response);
              }
            });
          } catch (e) {
            console.log('[ChessNextMove] Exception sending message:', e.message);
          }
        }
        return { success: true, fen: boardData.fen };
      }
      
      return { success: false, error: 'No chess board found on this page' };
    } catch (err) {
      console.error('[ChessNextMove] Scan error:', err);
      return { success: false, error: err.message };
    }
  }
  
  // Scan wc-chess-board web component
  function scanWcChessBoard() {
    const board = document.querySelector('wc-chess-board');
    if (!board) return null;
    
    const flipped = board.classList.contains('flipped');
    return parsePiecesBySquareClass(board, flipped);
  }
  
  // Scan chess-board web component
  function scanChessBoardComponent() {
    const board = document.querySelector('chess-board');
    if (!board) return null;
    
    const flipped = board.classList.contains('flipped');
    
    if (board.game && board.game.getFEN) {
      return { fen: board.game.getFEN(), flipped };
    }
    
    return parsePiecesBySquareClass(board, flipped);
  }
  
  // Scan cg-board (Lichess-style with transform positions)
  function scanCgBoard() {
    const cgBoard = document.querySelector('cg-board');
    console.log('[ChessNextMove] scanCgBoard - cgBoard found:', !!cgBoard);
    if (!cgBoard) return null;
    
    // Check if a piece is being dragged - SKIP if dragging
    const draggingPiece = cgBoard.querySelector('piece.dragging');
    if (draggingPiece) {
      console.log('[ChessNextMove] SKIP SCAN - piece is being dragged!');
      return null;
    }
    
    // Check for ghost piece (premove indicator)
    const ghostPiece = cgBoard.querySelector('piece.ghost');
    if (ghostPiece) {
      console.log('[ChessNextMove] SKIP SCAN - ghost piece detected (premove)');
      return null;
    }
    
    // Check if animation is in progress
    const animatingPiece = cgBoard.querySelector('piece.anim');
    if (animatingPiece) {
      console.log('[ChessNextMove] SKIP SCAN - piece animation in progress');
      return null;
    }
    
    const cgWrap = document.querySelector('.cg-wrap');
    const flipped = cgWrap?.classList.contains('orientation-black') || false;
    console.log('[ChessNextMove] scanCgBoard - flipped:', flipped);
    
    const board = Array(8).fill(null).map(() => Array(8).fill(null));
    const pieces = cgBoard.querySelectorAll('piece');
    
    console.log('[ChessNextMove] scanCgBoard - pieces found:', pieces.length);
    if (pieces.length === 0) return null;
    
    const boardRect = cgBoard.getBoundingClientRect();
    const squareSize = boardRect.width / 8;
    console.log('[ChessNextMove] scanCgBoard - boardRect:', boardRect.width, 'x', boardRect.height, 'squareSize:', squareSize);
    
    pieces.forEach((piece, idx) => {
      const classes = Array.from(piece.classList);
      
      let color = null;
      let type = null;
      
      for (const cls of classes) {
        if (cls === 'white' || cls === 'black') {
          color = cls;
        }
        if (PIECE_MAP_TYPES[cls]) {
          type = cls;
        }
      }
      
      if (!color || !type) {
        if (idx < 3) console.log('[ChessNextMove] scanCgBoard - piece', idx, 'classes:', classes, 'color:', color, 'type:', type);
        return;
      }
      
      const transform = piece.style.transform;
      const cgTransform = piece.style.cssText;
      // Handle negative values in transform
      const match = transform.match(/translate\((-?\d+(?:\.\d+)?)px,\s*(-?\d+(?:\.\d+)?)px\)/);
      
      if (idx < 3) console.log('[ChessNextMove] scanCgBoard - piece', idx, 'transform:', transform, 'match:', match);
      
      if (match) {
        let x = parseFloat(match[1]);
        let y = parseFloat(match[2]);
        
        let file = Math.round(x / squareSize);
        let rank = 7 - Math.round(y / squareSize);
        
        if (flipped) {
          file = 7 - file;
          rank = 7 - rank;
        }
        
        if (file >= 0 && file < 8 && rank >= 0 && rank < 8) {
          const pieceChar = PIECE_MAP_TYPES[type][color];
          board[7 - rank][file] = pieceChar;
        }
      }
    });
    
    const fen = boardToFen(board);
    console.log('[ChessNextMove] scanCgBoard - generated FEN:', fen);
    const turn = detectTurn(flipped);
    const castling = calculateCastlingRights(board);
    const fullFen = fen + ' ' + turn + ' ' + castling + ' - 0 1';
    
    return { fen: fullFen, flipped };
  }
  
  // Scan generic pieces with square-XY class pattern
  function scanGenericPieces() {
    const boardContainer = document.querySelector('.board') || 
                          document.querySelector('[class*="board"]') ||
                          document.querySelector('.chessboard');
    
    if (!boardContainer) return null;
    
    const flipped = boardContainer.classList.contains('flipped') ||
                   document.querySelector('.flipped') !== null;
    
    return parsePiecesBySquareClass(boardContainer, flipped);
  }
  
  // Parse pieces that use square-XY class naming
  function parsePiecesBySquareClass(boardEl, flipped) {
    const board = Array(8).fill(null).map(() => Array(8).fill(null));
    const pieces = boardEl.querySelectorAll('.piece');
    
    if (pieces.length === 0) return null;
    
    pieces.forEach(piece => {
      const classes = piece.className.split(' ');
      
      let pieceType = null;
      for (const cls of classes) {
        if (PIECE_MAP_CLASSES[cls]) {
          pieceType = PIECE_MAP_CLASSES[cls];
          break;
        }
      }
      
      let square = null;
      for (const cls of classes) {
        const match = cls.match(/square-(\d)(\d)/);
        if (match) {
          const file = parseInt(match[1]) - 1;
          const rank = parseInt(match[2]) - 1;
          square = { file, rank };
          break;
        }
      }
      
      if (pieceType && square) {
        board[7 - square.rank][square.file] = pieceType;
      }
    });
    
    const fen = boardToFen(board);
    const turn = detectTurn(flipped);
    const fullFen = fen + ' ' + turn + ' KQkq - 0 1';
    
    return { fen: fullFen, flipped };
  }
  
  // Calculate castling rights based on piece positions
  function calculateCastlingRights(board) {
    let castling = '';
    
    // White back rank is board[7] (rank 1)
    // Black back rank is board[0] (rank 8)
    
    // White Kingside (K): King on e1 (file 4) AND Rook on h1 (file 7)
    if (board[7][4] === 'K' && board[7][7] === 'R') {
      castling += 'K';
    }
    
    // White Queenside (Q): King on e1 (file 4) AND Rook on a1 (file 0)
    if (board[7][4] === 'K' && board[7][0] === 'R') {
      castling += 'Q';
    }
    
    // Black Kingside (k): King on e8 (file 4) AND Rook on h8 (file 7)
    if (board[0][4] === 'k' && board[0][7] === 'r') {
      castling += 'k';
    }
    
    // Black Queenside (q): King on e8 (file 4) AND Rook on a8 (file 0)
    if (board[0][4] === 'k' && board[0][0] === 'r') {
      castling += 'q';
    }
    
    return castling || '-';
  }
  
  // Convert board array to FEN
  function boardToFen(board) {
    const rows = [];
    
    for (let rank = 0; rank < 8; rank++) {
      let row = '';
      let empty = 0;
      
      for (let file = 0; file < 8; file++) {
        const piece = board[rank][file];
        if (piece) {
          if (empty > 0) {
            row += empty;
            empty = 0;
          }
          row += piece;
        } else {
          empty++;
        }
      }
      
      if (empty > 0) {
        row += empty;
      }
      
      rows.push(row);
    }
    
    return rows.join('/');
  }
  
  // Detect whose turn it is from UI clocks
  function detectTurn(isFlipped) {
    // Method 1: clock-player-turn class with color
    const whiteClockActive = document.querySelector('.clock-white.clock-player-turn, [class*="clock"][class*="white"][class*="turn"]');
    if (whiteClockActive) return 'w';
    
    const blackClockActive = document.querySelector('.clock-black.clock-player-turn, [class*="clock"][class*="black"][class*="turn"]');
    if (blackClockActive) return 'b';
    
    // Method 2: Bottom/top clock active
    const bottomClockActive = document.querySelector('.clock-bottom.clock-player-turn, .rclock-bottom.running');
    if (bottomClockActive) {
      return isFlipped ? 'b' : 'w';
    }
    
    const topClockActive = document.querySelector('.clock-top.clock-player-turn, .rclock-top.running');
    if (topClockActive) {
      return isFlipped ? 'w' : 'b';
    }
    
    // Method 3: Lichess running clock
    const runningClock = document.querySelector('.rclock.running');
    if (runningClock) {
      const isBottom = runningClock.classList.contains('rclock-bottom');
      return isBottom ? (isFlipped ? 'b' : 'w') : (isFlipped ? 'w' : 'b');
    }
    
    // Default
    return 'w';
  }
  
  // Cleanup on unload
  function cleanup() {
    if (observer) {
      observer.disconnect();
      observer = null;
    }
    if (scanInterval) {
      clearInterval(scanInterval);
      scanInterval = null;
    }
    if (debounceTimer) {
      clearTimeout(debounceTimer);
      debounceTimer = null;
    }
    // Clear storage flag
    try {
      chrome.storage.session.remove([TAB_KEY]);
    } catch (e) {
      // Ignore
    }
  }
  
  // Start
  console.log('[ChessNextMove] Document readyState:', document.readyState);
  if (document.readyState === 'loading') {
    console.log('[ChessNextMove] Document still loading, waiting for DOMContentLoaded...');
    document.addEventListener('DOMContentLoaded', () => {
      console.log('[ChessNextMove] DOMContentLoaded fired, calling init...');
      init();
    });
  } else {
    console.log('[ChessNextMove] Document already loaded, calling init immediately...');
    init();
  }
  
  window.addEventListener('unload', cleanup);
  console.log('[ChessNextMove] === CONTENT SCRIPT SETUP COMPLETE ===');
})();
