// ChessNextMove Extension - Background Service Worker
// Handles communication and programmatic script injection

// Store latest board state
let currentBoardState = {
  fen: null,
  flipped: false
};

// Store sidepanel port
let sidepanelPort = null;

// Track injected tabs to avoid duplicate injection
const injectedTabs = new Set();

// Open side panel when extension icon is clicked
chrome.action.onClicked.addListener(async (tab) => {
  await chrome.sidePanel.open({ tabId: tab.id });
  
  // Inject content script after panel opens
  setTimeout(() => {
    injectContentScript(tab.id);
  }, 300);
});

// Set side panel behavior
chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true });

// Inject content script into a tab
async function injectContentScript(tabId) {
  console.log('[Background] === INJECT CONTENT SCRIPT ===');
  console.log('[Background] Target tabId:', tabId);
  console.log('[Background] Already injected tabs:', Array.from(injectedTabs));
  
  try {
    // Check if tab exists and is valid
    const tab = await chrome.tabs.get(tabId);
    console.log('[Background] Tab URL:', tab.url);
    
    // Skip chrome:// and other restricted URLs
    if (!tab.url || tab.url.startsWith('chrome://') || tab.url.startsWith('chrome-extension://') || tab.url.startsWith('about:')) {
      console.log('[Background] SKIP: Restricted URL');
      return;
    }
    
    // Check if already injected
    if (injectedTabs.has(tabId)) {
      console.log('[Background] SKIP: Already injected, sending triggerScan instead');
      chrome.tabs.sendMessage(tabId, { type: 'triggerScan' }).catch((err) => {
        console.log('[Background] triggerScan message failed:', err.message);
      });
      return;
    }
    
    // Inject the universal content script
    console.log('[Background] Executing script injection...');
    await chrome.scripting.executeScript({
      target: { tabId: tabId },
      files: ['content-universal.js']
    });
    
    injectedTabs.add(tabId);
    console.log('[Background] SUCCESS: Injected content script into tab:', tabId);
    
    // Request immediate scan
    setTimeout(() => {
      console.log('[Background] Sending scanBoard message to tab...');
      chrome.tabs.sendMessage(tabId, { type: 'scanBoard' }).catch((err) => {
        console.log('[Background] scanBoard message failed (script may not be ready):', err.message);
      });
    }, 200);
    
  } catch (err) {
    console.log('[Background] INJECTION FAILED:', err.message);
  }
}

// Listen for connections from sidepanel
chrome.runtime.onConnect.addListener((port) => {
  console.log('[Background] === CONNECTION RECEIVED ===' , 'port.name:', port.name);
  
  if (port.name === 'sidepanel') {
    sidepanelPort = port;
    console.log('[Background] Sidepanel connected');
    
    // Send current board state if available
    if (currentBoardState.fen) {
      console.log('[Background] Sending cached board state:', currentBoardState.fen);
      port.postMessage({
        type: 'boardUpdate',
        data: currentBoardState
      });
    } else {
      console.log('[Background] No cached board state to send');
    }
    
    // Inject script into current tab when panel connects
    console.log('[Background] Querying active tab for injection...');
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      console.log('[Background] Active tabs found:', tabs.length);
      if (tabs[0]) {
        console.log('[Background] Injecting into active tab:', tabs[0].id, tabs[0].url);
        injectContentScript(tabs[0].id);
      } else {
        console.log('[Background] WARNING: No active tab found!');
      }
    });
    
    port.onDisconnect.addListener(() => {
      console.log('[Background] Sidepanel disconnected');
      sidepanelPort = null;
    });
    
    // Listen for messages from sidepanel
    port.onMessage.addListener((msg) => {
      console.log('[Background] Message from sidepanel:', msg.type);
      
      if (msg.type === 'requestScan') {
        console.log('[Background] Scan requested, querying active tab...');
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
          if (tabs[0]) {
            console.log('[Background] Triggering injection for tab:', tabs[0].id);
            injectContentScript(tabs[0].id);
          } else {
            console.log('[Background] WARNING: No active tab for scan!');
          }
        });
      }
    });
  }
});

// Listen for messages from content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('[Background] === MESSAGE RECEIVED ===', 'type:', message.type, 'from:', sender.tab?.id || 'unknown');
  
  if (message.type === 'boardDetected') {
    console.log('[Background] Board detected! FEN:', message.fen);
    currentBoardState = {
      fen: message.fen,
      flipped: message.flipped
    };
    
    // Forward to sidepanel
    if (sidepanelPort) {
      console.log('[Background] Forwarding board state to sidepanel');
      sidepanelPort.postMessage({
        type: 'boardUpdate',
        data: currentBoardState
      });
    } else {
      console.log('[Background] WARNING: No sidepanel port to forward board state!');
    }
    
    sendResponse({ success: true });
  }
  
  if (message.type === 'boardCleared') {
    console.log('[Background] Board cleared');
    currentBoardState = {
      fen: null,
      flipped: false
    };
    
    if (sidepanelPort) {
      sidepanelPort.postMessage({
        type: 'boardCleared'
      });
    }
    
    sendResponse({ success: true });
  }
  
  // Force reinject - clear all tracked tabs so content script gets freshly injected
  if (message.type === 'forceReinject') {
    console.log('[Background] === FORCE REINJECT REQUESTED ===');
    console.log('[Background] Clearing injectedTabs set, was:', Array.from(injectedTabs));
    injectedTabs.clear();
    console.log('[Background] injectedTabs cleared, now:', Array.from(injectedTabs));
    
    // Clear chrome.storage.session to allow fresh injection
    console.log('[Background] Clearing chrome.storage.session...');
    chrome.storage.session.clear().then(() => {
      console.log('[Background] SUCCESS: Session storage cleared');
    }).catch(err => {
      console.log('[Background] ERROR clearing session storage:', err.message);
    });
    
    // Also clear the current board state
    console.log('[Background] Clearing cached board state');
    currentBoardState = { fen: null, flipped: false };
    
    sendResponse({ success: true });
  }
  
  // Request scan from content script (when already injected)
  if (message.type === 'requestScanFromContent') {
    console.log('[Background] Content script requested scan relay');
    // Just trigger a scan on the current tab
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs[0]) {
        console.log('[Background] Sending triggerScan to tab:', tabs[0].id);
        chrome.tabs.sendMessage(tabs[0].id, { type: 'triggerScan' }).catch((err) => {
          console.log('[Background] triggerScan failed:', err.message);
        });
      }
    });
    sendResponse({ success: true });
  }
  
  return true;
});

// Inject script when tab updates (page navigation)
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (changeInfo.status === 'complete' && sidepanelPort) {
    // Clear from injected set so it can be re-injected
    injectedTabs.delete(tabId);
    
    setTimeout(() => {
      injectContentScript(tabId);
    }, 500);
  }
});

// Inject script when user switches tabs
chrome.tabs.onActivated.addListener((activeInfo) => {
  if (sidepanelPort) {
    setTimeout(() => {
      injectContentScript(activeInfo.tabId);
    }, 200);
  }
});

// Clean up when tab closes
chrome.tabs.onRemoved.addListener((tabId) => {
  injectedTabs.delete(tabId);
});
